<?php

/**
* Main class ThimbleParser
* @author Rudolf Naprstek
* @package thimble_doc
* @version 1.0
*/

require ("../views/view_api.php");

/**
 * include (T_ABSTRACT);
 * include $neco;
*/

/// Main parser class, which has all the kids
 class ThimbleParser {
    
    public $code;

    /// index of the current token
    public $current_token = 0;
    
    /// total number of tokens in code
    public $total_tokens = 0;
    
    /// tokens of the code
    public $tokens = array();
    
    /// result array with classes
    public $result = array();
    
    public $ignore_first_definition = false;

    /// ignored tokens between comment and class/function/method/var etc ...
    static $ignored_comment_tokens = array(T_WHITESPACE, T_COMMENT, T_DOC_COMMENT, T_PRIVATE, T_PROTECTED, T_PUBLIC, T_STATIC, T_ABSTRACT, T_FINAL);
    
    /// tokens to watch out for before class/function/variable... definition
    static $remember_tokens = array(T_PRIVATE, T_PROTECTED, T_PUBLIC, T_STATIC, T_ABSTRACT, T_COMMENT, T_DOC_COMMENT, T_FINAL);
    
    function __construct($code, $ignore_first_definition = false) {
      $this->code = $code;
      $this->current_token = 0;
      $this->tokens = token_get_all($this->code);
      $this->total_tokens = count($this->tokens);
      $this->ignore_first_definition = $ignore_first_definition;
     // $this->print_tokens();
     // die();
    }
    
    function parse() {
      
      $running = true;
      
      $tokens_before_declaration = array();
      
      $ignore_first_definition = $this->ignore_first_definition;
      
      // it's array so we can use it in ThimbleParserBase by method parse_comments
      $comment_token = array();
      
      $main_comment_area = true;
      $tmp_comment = false;
      
      while ($running) {
        
        // This condition is for parsing main comment at the beginning of the file
        if ($main_comment_area && is_array($this->tokens[$this->current_token]) && !$ignore_first_definition) {
          if ($this->tokens[$this->current_token][0] == T_COMMENT || $this->tokens[$this->current_token][0] == T_DOC_COMMENT) {
            if (!$tmp_comment) {
              $tmp_comment = $this->tokens[$this->current_token];
            } else {
              if (count($tokens_before_declaration)==0) $tokens_before_declaration[] = $tmp_comment;
              $child = new ThimbleParserMainComment($this->tokens, $this->current_token, $tokens_before_declaration);
              $this->result[] = $child->parse();
              $main_comment_area = false;
              $tokens_before_declaration = array();
              $tmp_comment = false;
            }
          }
        }
        
        if ($this->tokens[$this->current_token][0] == T_COMMENT || $this->tokens[$this->current_token][0] == T_DOC_COMMENT) {
          $comment_token[] = $this->tokens[$this->current_token][1];
        } elseif ($this->tokens[$this->current_token][0] == T_CLASS) {
          $main_comment_area = false;
          if ($ignore_first_definition) {
            $ignore_first_definition = false; 
          } else {
            $child = new ThimbleParserClass($this->tokens, $this->current_token, $tokens_before_declaration);
            $this->result[] = $child->parse();
            $this->current_token = $child->current_token-1;
          }
        } elseif ($this->tokens[$this->current_token][0] == T_FUNCTION) {
          $main_comment_area = false;
          if ($ignore_first_definition) {
            $ignore_first_definition = false; 
          } else {
            $child = new ThimbleParserFunction($this->tokens, $this->current_token, $tokens_before_declaration);
            $this->result[] = $child->parse();
            $this->current_token = $child->current_token-1;
          }
        } elseif ($this->tokens[$this->current_token][0] == T_REQUIRE || 
           $this->tokens[$this->current_token][0] == T_REQUIRE_ONCE || 
           $this->tokens[$this->current_token][0] == T_INCLUDE ||
           $this->tokens[$this->current_token][0] == T_INCLUDE_ONCE) {
           
           $include_type = $this->tokens[$this->current_token][1];
             
           $run_token_search = true;
           while ($run_token_search) {
             if (is_array($this->tokens[$this->current_token]) && (
                  $this->tokens[$this->current_token][0] == T_CONSTANT_ENCAPSED_STRING ||
                  $this->tokens[$this->current_token][0] == T_STRING ||
                  $this->tokens[$this->current_token][0] == T_VARIABLE)) {
               $run_token_search = false;

               if (!isset($child_includes)) {
                 $child_includes = new ThimbleParserIncludes($this->tokens, $this->current_token, $tokens_before_declaration);
                 $this->result[] = $child_includes;
               } 
               $child_includes->includes[] = array('type' => $include_type, 'file' => trim($this->tokens[$this->current_token][1],"'\""));
               
             }
             $this->current_token++;
             if ($this->current_token >= $this->total_tokens) $run_token_search = false;
           } 
        }


        
        // comments are only accepted, when they are just before function, class, etc... between them can be only whitespaces
        if (!in_array($this->tokens[$this->current_token][0], ThimbleParser::$ignored_comment_tokens)){
          $comment_token = array();
          $tokens_before_declaration = array();
        } elseif (in_array($this->tokens[$this->current_token][0], ThimbleParser::$remember_tokens)) {
          $tokens_before_declaration[] = $this->tokens[$this->current_token];
        }
        
        $this->current_token++;
        
        if ($this->current_token >= $this->total_tokens) {
          $running = false;
        }
        
      }
      
    }
    
    /// only test function
    function print_tokens() {
      for ($i=0;$i<$this->total_tokens;$i++) {
        echo $i.' - ';
        if (is_array($this->tokens[$i])) {
          echo token_name($this->tokens[$i][0]).'-'.trim(htmlentities($this->tokens[$i][1]))."-\n";
        } else {
          echo $this->tokens[$i]."\n";
        }
      }
      
    }
    
  }
  
  /// General class for other token classes - class, function, etc ...
  
  class ThimbleParserBase {
    
    public $tokens = array();
    public $total_tokens = 0;
    public $current_token = 0;
    public $tokens_before_declaration;
    public $info_allowed_tags = array('author', 'copyright', 'deprecated', 'license', 'package',
                                      'param', 'return', 'since', 'todo', 'tutorial', 'var', 'version');
    
    /// what tokens are allowed for info, e.g. public, static, private, etc ...
    public $allowed_tokens_before_definition = array();
                                      
    /// information about class/function/method/variable etc...
    public $name;
    public $comment_text;
    public $info = array();
    /// these can be - public, static, private, abstract, etc ...
    public $properties = array();
    /// other ThimbleParser objects which are "kids" of current class
    public $kids = array();
    
    function __construct($tokens, $current_token, $tokens_before_declaration) {
      $this->tokens = $tokens;
      $this->current_token = $current_token;
      $this->total_tokens = count($tokens);
      $this->tokens_before_declaration = $tokens_before_declaration;
      $this->parse_comments();
      
      // get properties of function/class
      foreach ($this->tokens_before_declaration as $token) {
        if (in_array($token[0], $this->allowed_tokens_before_definition)) {
          //$this->info['properties'][] = $token[1];
          $this->properties[] = $token[1];
        }
      }
    }
    
    /// Parses comments from $tokens_before_declararion and puts them into $this->comments;    
    function parse_comments() {
      $regs = array();
      foreach ($this->tokens_before_declaration as $token) {
        $token[1] = trim($token[1]);
        if ($token[0] == T_DOC_COMMENT || (($token[0] == T_COMMENT) && (preg_match('/^\/\*/uU', $token[1])))) {
          $comments = explode("\n", $token[1]);
          if (strpos(array_shift($comments), '/*')!==false) {
            array_pop($comments);
            foreach ($comments as $comment) {
              if (preg_match('/^[ ]*\*[ ]+(.+)$/uU', $comment, $regs)) {
                $comment = $this->parse_comment_line($regs[1]);
                if ($comment!='') {
                  if ($this->comment_text!='') $this->comment_text .= "\n";
                  $this->comment_text .= $comment;
                }
              }
            }
          }
        } elseif ($token[0] == T_COMMENT) {
          $comment = $token[1];
          if (preg_match('/^\/\/\/[ ]+(.+)$/uU', $comment, $regs)) {
            $comment = $this->parse_comment_line($regs[1]);
            if ($comment!='') {
              if ($this->comment_text!='') $this->comment_text .= "\n";
              $this->comment_text .= $comment;
            }
          }
        }
      }
      
    }
    
    function parse_comment_line($comment) {
      $regs = array();
      if (preg_match('/^@([a-zA-Z]+) (.*)$/uU', $comment, $regs)) {
        if (in_array($regs[1], $this->info_allowed_tags)) {
          $this->info[] = array('info_name' => $regs[1], 'info_value' => $regs[2]);
          $comment = '';
        }
      }
      return $comment;
    }
    
    /// Walks through tokens, until specified token is found
    function go_until_token($token, $is_character = false, $return_inside_text = false, $escape_character = false) {
      
      $inside_text = '';
      
      $depth = 0;
      
      while (true) {
        
        if ($escape_character && $this->tokens[$this->current_token] == $escape_character) {
          $depth++;
        }
                
        if (!$is_character && $this->tokens[$this->current_token][0] == $token) {
          if ($return_inside_text) {
            return $inside_text; 
          } else {
            return true;
          }
        } elseif ($is_character && $this->tokens[$this->current_token] == $token && $depth == 0) {
          if ($return_inside_text) {
            return $inside_text; 
          } else {
            return true;
          }
        } else {
          if (is_array($this->tokens[$this->current_token])) {
            $inside_text .= $this->tokens[$this->current_token][1];
          } else {
            $inside_text .= $this->tokens[$this->current_token];
          }
          $this->current_token++;
        }
        
        if ($depth > 0 && isset($this->tokens[$this->current_token]) && $this->tokens[$this->current_token] == $token) {
          $depth--; 
        }
        
        if ($this->current_token >= $this->total_tokens) {
          return false;
        }
      }
      
    }
       
  }
  
  
  class ThimbleParserMainComment extends ThimbleParserBase {
    
    function parse() {
      return $this;
    }
    
  }

  
  class ThimbleParserIncludes extends ThimbleParserBase {
    
    public $includes = array();
    
    function parse() {
      return $this;
    }
    
    function parse_comments() { }
    
  }
  
  class ThimbleParserClass extends ThimbleParserBase {
    
    /// what tokens are allowed for info, e.g. public, static, private, etc ...
    public $allowed_tokens_before_definition = array(T_ABSTRACT);
    
    /// what tokens can precede variable
    public $allowed_tokens_before_variable = array(T_PUBLIC, T_PRIVATE, T_PROTECTED);
    
    public $variables = array();
    
    function parse() {
      
      $depth = -999;
      $running = true;
      $inside_text = '';
      
      $token_before_variable = '';
      $comments_before_variable = '';
      
      $regs = array();
      
      if ($this->go_until_token(T_STRING)) {
        $this->name = $this->tokens[$this->current_token][1];
        
        $current_token = $this->current_token;
        
        while ($running) {
          
          if (!is_array($this->tokens[$current_token]) && $this->tokens[$current_token] =='{') {
            if ($depth == -999) { 
              $depth = 1;
            } else {
              $depth++;
            }
          }
          
          if (!is_array($this->tokens[$current_token]) && $this->tokens[$current_token] =='}') {
            $depth--;
          }
          
          if (is_array($this->tokens[$current_token]) && ($this->tokens[$current_token][0]==T_COMMENT || $this->tokens[$current_token][0]==T_DOC_COMMENT)) {
            $comments_before_variable[] = $this->tokens[$current_token][1];
          }
          

          if (is_array($this->tokens[$current_token]) && in_array($this->tokens[$current_token][0], $this->allowed_tokens_before_variable)) {
            $token_before_variable = $this->tokens[$current_token][1];
          }
          
          if (is_array($this->tokens[$current_token]) && $this->tokens[$current_token][0]==T_VARIABLE && $token_before_variable!='') {
            $comments = '';
            foreach ($comments_before_variable as $comment) {
            
              if (preg_match('/^\/\/\/[ ]+(.+)$/uU', trim($comment), $regs)) {
                $comments .= $this->parse_comment_line($regs[1]);
              } elseif (preg_match('/^\/\*\*.*$/uU', trim($comment))) {
                $comments_array = explode("\n", trim($token[1]));
                if (array_shift($comments_array)=='/**') {
                  array_pop($comments_array);
                  foreach ($comments_array as $comment2) {
                    if (preg_match('/^\*[ ]+(.+)$/uU', $comment2, $regs)) {
                      $comments .= $this->parse_comment_line($regs[1]);
                    }
                  }
                }
              }
            }
            $this->variables[] = array('name' => $this->tokens[$current_token][1], 'type' => $token_before_variable, 'comment' => $comments);
            $comments_before_variable = array();
            $token_before_variable = '';
          }
          
          if (is_array($this->tokens[$current_token]) && !in_array($this->tokens[$current_token][0], ThimbleParser::$ignored_comment_tokens)){
            $comments_before_variable = array();
          }
          
          if (is_array($this->tokens[$current_token])) {
            $inside_text .= $this->tokens[$current_token][1];
          } else {
            $inside_text .= $this->tokens[$current_token];
          }
          
          if ($depth == 0) $running = false;
          
          $current_token++;
          
          if ($current_token == $this->total_tokens) {
            $running = false;
          }
          
        }
        
        $inside_text = '<?php class '.$inside_text.' ?>';
        
        $class_parser = new ThimbleParser($inside_text, true);
        $class_parser->parse();
        $this->kids = $class_parser->result;
        
        $this->current_token = $current_token;
      } else {
        return false;
      }
      
      
      return $this;
      
    }
    
  }
  
  
  
 class ThimbleParserFunction extends ThimbleParserBase {
    
    /// arguments of the function
    public $arguments;
    /// what tokens are allowed for info, e.g. public, static, private, etc ...
    public $allowed_tokens_before_definition = array(T_PUBLIC, T_STATIC, T_PRIVATE, T_PROTECTED, T_ABSTRACT);
    
    public function parse() {
      
      $depth = -999;
      $running = true;
      $inside_text = '';
      
      if ($this->go_until_token(T_STRING)) {
        $this->name = $this->tokens[$this->current_token][1];
        
        // get arguments of the function
        $this->go_until_token('(', true);
        if ($arguments = $this->go_until_token(')', true, true, '(')) {
          $regs = array(); $regs2 = array(); $regs3 = array();
          preg_match('/^\((.*)$/uU', trim($arguments), $regs);
          foreach (explode(',', $regs[1]) as $argument) {
            if (preg_match('/^([&]?)(\$[a-zA-Z0-9_]+)[ ]*=?[ ]*(.*)/u', trim($argument), $regs2)) {
              $comment = '';
              foreach ($this->info as $key => $info) {
                if ($info['info_name']=='param' && preg_match('/^\\'.$regs2[2].'[ ](.*)/u', $info['info_value'], $regs3)) {
                  $comment = $regs3[1];
                  unset($this->info[$key]);
                }
              }
              $this->arguments[] = array('name' => $regs2[2], 'default_value' => $regs2[3], 'passed_by_reference' => $regs2[1], 'comment' => $comment);
            }
          }
        }
        
        $current_token = $this->current_token;
        
        if (!in_array('abstract', $this->properties)) {
        
          while ($running) {
            
            if (!is_array($this->tokens[$current_token]) && $this->tokens[$current_token] =='{') {
              if ($depth == -999) { 
                $depth = 1;
              } else {
                $depth++;
              }
            }
          
            if (!is_array($this->tokens[$current_token]) && $this->tokens[$current_token] =='}') {
              $depth--;
            }
          
            if (is_array($this->tokens[$current_token])) {
              $inside_text .= $this->tokens[$current_token][1];
            } else {
              $inside_text .= $this->tokens[$current_token];
            }
          
            if ($depth == 0) $running = false;
          
            $current_token++;
          
            if ($current_token == $this->total_tokens) {
              $running = false;
            }
          
          }

          $inside_text = '<?php function '.$inside_text.' ?>';
          $func_parser = new ThimbleParser($inside_text, true);
          $func_parser->parse();
          $this->kids = $func_parser->result;
        
          $this->current_token = $current_token;
        }
      } else {
        return false;
      }
      
      
      return $this;
      
    }
    
  }
  
  
  /// This is comment of test3 class
  /// @since 0.1
  
  abstract class test3 {
    
  }
  
  /// Some test function
  /// @return nothing :-)
  /// @param $arg1 First argument
  /// @param $arg2 Second argument
  /// @since 0.5 version 
  
  function test2($arg1, &$arg2 = 'xx')  {
    require ("views/view_api.php");
    include_once('views/view_api.php');
  }

  
  
?>